Namespace linq

#rem monkeydoc @hidden
#end
Struct LinqHelper<TCollection,T>

Public

	Function CreateForList<T>:LinqHelper<List<T>,T>()
		Return New LinqHelper<List<T>,T>()
	End

	Function CreateForStack<T>:LinqHelper<Stack<T>,T>()
		Return New LinqHelper<Stack<T>,T>()
	End
	
Private
	
	Alias CollectionSelector<T,TResultCollection>:TResultCollection(item:T)
	Alias CollectionSelectorWithIndex<T,TResultCollection>:TResultCollection(item:T, index:Int)

	Method New:Void()
	End

Public
	
	Method Aggregate:T(collection:TCollection, seed:T, accumulator:Accumulator<T>)

		If collection.Empty Then Return seed

		Local accumulation := seed

		For Local item := EachIn collection.All()
			accumulation = accumulator(accumulation, item)
		End
		
		Return accumulation

	End
	
	Method All:Bool(collection:TCollection, predicate:Predicate<T>)

		If collection.Empty Then Return False

		For Local item := EachIn collection.All()
			If Not predicate(item) Then Return False
		End
		
		Return True

	End

	Method Any:Bool(collection:TCollection)
		Return Not collection.Empty
	End
	
	Method Any:Bool(collection:TCollection, predicate:Predicate<T>)

		If collection.Empty Then Return False

		For Local item := EachIn collection.All()
			If predicate(item) Then Return True
		End
		
		Return False

	End

	Method Average:T(collection:TCollection, transform:NumericTransform<T>) Where T Extends INumeric

		' Floating point or integer type?
		Local value:T = 1.5
		If String(value).Contains(".")
			Return CalculateAverage<Double>(collection, transform)
		Else
			Return CalculateAverage<Long>(collection, transform)
		End

	End

	Method Concat:TCollection(collection:TCollection, secondCollection:TCollection)
	
		Local newCollection := New TCollection()
		
		For Local item := EachIn collection.All()
			newCollection.Add(item)
		End

		For Local item := EachIn secondCollection.All()
			newCollection.Add(item)
		End

		Return newCollection
		
	End
	
	Method Contains:Bool(collection:TCollection, match:T)
		
		Return Any(collection, Lambda:Bool(item:T)
			Return item = match
		End)
		
	End
	
	Method Contains:Bool(collection:TCollection, match:T, comparer:Comparer<T>)
	
		If comparer = Null Then Return collection.Contains(match)

		If collection.Empty Then Return False
	
		For Local item := EachIn collection.All()
			If comparer(match, item) Then Return True
		End
		
		Return False
	
	End	

	Method Count:UInt(collection:TCollection, predicate:Predicate<T>)
	
		If collection.Empty Then Return 0
		
		Local count:UInt = 0

		For Local item := EachIn collection.All()
			If predicate(item) Then count += 1
		End
		
		Return count
		
	End
	
	Method DefaultIfEmpty:TCollection(collection:TCollection, defaultValue:T)
		
		If Not collection.Empty Then Return collection
		
		Local newCollection:= New TCollection()
		newCollection.Add(defaultValue)
		Return newCollection
		
	End
	
	Method Distinct:TCollection(collection:TCollection, comparer:Comparer<T>)
	
		Local newCollection := New TCollection()

		If collection.Empty Then Return newCollection	
	
		For Local item := EachIn collection.All()
			If Not Contains(newCollection, item, comparer) Then newCollection.Add(item)
		End
		
		Return newCollection
	
	End
	
	Method ElementAtOrDefault:T(collection:TCollection, index:UInt, defaultValue:T)
		
		If collection.Empty Or index >= collection.Count() Then Return defaultValue
		
		Local count := 0
		
		For Local item := EachIn collection.All()
			If count = index Then Return item
			count += 1
		End
		
		Return defaultValue ' shouldn't get here
		
	End
	
	Method Except:TCollection(collection:TCollection, secondCollection:TCollection, comparer:Comparer<T>)
	
		Local newCollection := New TCollection()
	
		For Local item := Eachin collection.All()
			If Not Contains(secondCollection, item, comparer) Then newCollection.Add(item)
		End
		
		Return newCollection
		
	End
	
	Method FirstOrDefault:T(collection:TCollection, first:T, defaultValue:T)
		
		If collection.Empty Or first = Null
			Return defaultValue
		Else
			Return first
		End
		
	End

	Method FirstOrDefault:T(collection:TCollection, defaultValue:T, predicate:Predicate<T>)
	
		If collection.Empty Then Return defaultValue
		
		For Local item := EachIn collection.All()
			If predicate(item) Then Return item
		End
		
		Return defaultValue
	
	End
	
	Method Intersect:TCollection(collection:TCollection, secondCollection:TCollection, comparer:Comparer<T>)
		
		Local newCollection := New TCollection()
		If collection.Empty Or secondCollection.Empty Then Return newCollection
		
		For Local item := EachIn collection.All()
			If Contains(secondCollection, item, comparer) Then newCollection.Add(item)
		End
		
		Return newCollection
		
	End	

	Method Join<T,TInnerCollection,TInner,TKey,TResultCollection,TResult>:TResultCollection(outer:TCollection, inner:TInnerCollection, outerKeySelector:Selector<T,TKey>, innerKeySelector:Selector<TInner,TKey>, resultSelector:TResult(outer:T, inner:TInner))
		Return Join<T,TInnerCollection,TInner,TKey,TResultCollection,TResult>(outer, inner, outerKeySelector, innerKeySelector, resultSelector, Lambda:Bool(outer:TKey, inner:TKey)
			Return outer = inner
		End)
	End
	
	Method Join<T,TInnerCollection,TInner,TKey,TResultCollection,TResult>:TResultCollection(outer:TCollection, inner:TInnerCollection, outerKeySelector:Selector<T,TKey>, innerKeySelector:Selector<TInner,TKey>, resultSelector:TResult(outer:T, inner:TInner), keyComparer:Comparer<TKey>)

		' Build outer keys
		Local outerKeys := New Map<TKey, T>()
		For Local outerItem := EachIn outer.All()
			outerKeys.Add(outerKeySelector(outerItem), outerItem)
		End

		' Build inner keys
		Local innerKeys := New Map<TKey, TInner>()
		For Local innerItem := EachIn inner.All()
			innerKeys.Add(innerKeySelector(innerItem), innerItem)
		End
		
		Local results := New TResultCollection()
		
		' Find matching keys
		For Local outerKey := EachIn outerKeys.Keys.All()
	
			For Local innerKey := EachIn innerKeys.Keys.All()

				Local match := false

				If keyComparer <> Null
					match = keyComparer(outerKey, innerKey)
				Else
					match = outerKey = innerKey
				End
				
				If match
					results.Add(resultSelector(outerKeys.Get(outerKey), innerKeys.Get(innerKey)))
				End

			End
			
		End
		
		Return results
		
	End	

	Method LastOrDefault:T(collection:TCollection, last:T, defaultValue:T)
		
		If collection.Empty Or last = Null
			Return defaultValue
		Else
			Return last
		End
		
	End

	Method LastOrDefault:T(collection:TCollection, defaultValue:T, predicate:Predicate<T>)
	
		If collection.Empty Then Return defaultValue
		
		For Local item := EachIn collection.Backwards()
			If predicate(item) Then Return item
		End
		
		Return defaultValue
	
	End
	
	Method Max:T(collection:TCollection, transform:NumericTransform<T>) Where T Extends INumeric

		' Floating point or integer type?
		Local value:T = 1.5
		If String(value).Contains(".")
			Return CalculateMax<Double>(collection, transform, -1.7976931348623157E+308)
		Else
			Return CalculateMax<Long>(collection, transform, -9223372036854775808)
		End

	End		

	Method Min:T(collection:TCollection, transform:NumericTransform<T>) Where T Extends INumeric

		' Floating point or integer type?
		Local value:T = 1.5
		If String(value).Contains(".")
			Return CalculateMin<Double>(collection, transform, 1.7976931348623157E+308)
		Else
			Return CalculateMin<Long>(collection, transform, 9223372036854775807)
		End

	End	
	
	Method OrderBy:TCollection(collection:TCollection, sorter:Sorter<T>)
		
		Local sorted := New TCollection(collection)
		sorted.Sort(sorter)
		Return sorted
		
	End	
	
	Method OrderByDescending:TCollection(collection:TCollection, sorter:Sorter<T>)

		Local sorted := OrderBy(collection, sorter)
		
		Local descending := New TCollection()
		
		For Local item := EachIn sorted.Backwards()
			descending.Add(item)
		End
		
		Return descending
		
	End		
	
	Method Query:TCollection(collection:TCollection, predicate:Predicate<T>)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection

		For Local item := EachIn collection.All()
			If predicate(item) Then newCollection.Add(item)
		End
		
		Return newCollection
		
	End

	Method Query:TCollection(collection:TCollection, predicate:PredicateWithIndex<T>)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection
		
		Local index:UInt = 0

		For Local item := EachIn collection.All()
			If predicate(item, index) Then newCollection.Add(item)
			index += 1
		End
		
		Return newCollection
		
	End

	Function Range:TCollection(start:T, rate:T, count:UInt) Where T Extends INumeric
		
		Local newCollection := New TCollection()
		Local value:T = start
		
		For Local i := 0 To count -1
			newCollection.Add(value)
			value += rate
		End
		
		Return newCollection
		
	End
	
	Function Repeated:TCollection(item:T, count:UInt)
		
		Local collection := New TCollection()
		
		For Local i := 0 To count -1
			collection.Add(item)
		End
		
		Return collection
		
	End	

	Method Reverse:TCollection(collection:TCollection)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection
		
		For Local item := EachIn collection.Backwards()
			newCollection.Add(item)
		End
		
		Return newCollection
		
	End
		
	Method Select2<T,TResultCollection,TResult>:TResultCollection(collection:TCollection, selector:Selector<T,TResult>)
		
		Local newCollection := New TResultCollection()
		If collection.Empty Then Return newCollection
		
		For Local item := EachIn collection.All()
			newCollection.Add(selector(item))
		End
		
		Return newCollection
		
	End		

	Method Select2<T,TResultCollection,TResult>:TResultCollection(collection:TCollection, selector:SelectorWithIndex<T,TResult>)
		
		Local newCollection := New TResultCollection()
		If collection.Empty Then Return newCollection

		Local index:UInt = 0
		
		For Local item := EachIn collection.All()
			newCollection.Add(selector(item, index))
			index += 1
		End
		
		Return newCollection
		
	End

	Method SelectMany<T,TResultCollection>:TResultCollection(collection:TCollection, selector:CollectionSelector<T,TResultCollection>)
		
		Local newCollection := New TResultCollection()
		If collection.Empty Then Return newCollection
		
		For Local item := EachIn collection.All()
			For Local subItem := Eachin selector(item)
				newCollection.Add(subItem)
			End
		End
		
		Return newCollection
		
	End	

	Method SelectMany<T,TResultCollection>:TResultCollection(collection:TCollection, selector:CollectionSelectorWithIndex<T,TResultCollection>)
		
		Local newCollection := New TResultCollection()
		If collection.Empty Then Return newCollection
		
		Local index:UInt = 0
		
		For Local item := EachIn collection.All()
			For Local subItem := Eachin selector(item, index)
				newCollection.Add(subItem)
			End
			index += 1
		End
		
		Return newCollection
		
	End	

	Method SelectMany<T,TIntermediateCollection,TIntermediate,TResultCollection,TResult>:TResultCollection(collection:TCollection, collectionSelector:CollectionSelector<T,TIntermediateCollection>, resultSelector:Transform<T,TIntermediate,TResult>)
		
		Local newCollection := New TResultCollection()
		If collection.Empty Then Return newCollection
		
		For Local item := EachIn collection.All()
			For Local subItem := Eachin collectionSelector(item)
				newCollection.Add(resultSelector(item, subItem))
			End
		End
		
		Return newCollection
		
	End	

	Method SelectMany<T,TIntermediateCollection,TIntermediate,TResultCollection,TResult>:TResultCollection(collection:TCollection, collectionSelector:CollectionSelectorWithIndex<T,TIntermediateCollection>, resultSelector:Transform<T,TIntermediate,TResult>)
		
		Local newCollection := New TResultCollection()
		If collection.Empty Then Return newCollection
		
		Local index:UInt = 0
		
		For Local item := EachIn collection.All()
			For Local subItem := Eachin collectionSelector(item, index)
				newCollection.Add(resultSelector(item, subItem))
			End
			index += 1
		End
		
		Return newCollection
		
	End	

	Method Single:T(name:String, collection:TCollection, count:Int, first:T)
		
		If count = 1
			Return first
		Else
			Throw New Throwable() ' LinqException(name + " contains " + String(count) + " items")
		End
		
	End

	Method Single:T(collection:TCollection, predicate:Predicate<T>, getFirstMatch:T(matches:TCollection))
		
		Local matches := Query(collection, predicate)
		
		If matches.Count() = 1
			Return getFirstMatch(matches)
		Else
			Throw New Throwable() ' LinqException(matches.Count() + " matches found")
		End

	End

	Method SingleOrDefault:T(name:String, collection:TCollection, defaultValue:T, count:Int, first:T)
		
		If count = 1
			Return first
		Else
			If count = 0
				Return defaultValue
			Else
				Throw New Throwable() ' LinqException(name + " contains " + String(count) + " items")
			End
		End	
		
	End

	Method SingleOrDefault:T(collection:TCollection, defaultValue:T, predicate:Predicate<T>, getFirstMatch:T(matches:TCollection))
	
		Local matches := Query(collection, predicate)
		
		If matches.Count() = 1
			Return getFirstMatch(matches)
		Else
			If matches.Count() = 0
				Return defaultValue
			Else
				Throw New Throwable() ' LinqException(matches.Count() + " matches found")
			End
		End
	
	End

	Method Skip:TCollection(collection:TCollection, count:Int)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection

		Local index:UInt = 1
		
		For Local item := EachIn collection.All()
			If index > count Then newCollection.Add(item)
			index += 1
		End
		
		Return newCollection
		
	End
	
	Method SkipWhile:TCollection(collection:TCollection, predicate:Predicate<T>)
		Return SkipWhile(collection, Lambda:Bool(item:T, index:Int)
			Return predicate(item)
		End)
	End

	Method SkipWhile:TCollection(collection:TCollection, predicate:PredicateWithIndex<T>)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection

		Local skipping:Bool = True
		Local index:UInt = 0
		
		For Local item := EachIn collection.All()
		
			If skipping 
				If Not predicate(item, index)
					skipping = False
					newCollection.Add(item)
				End
			Else
				newCollection.Add(item)
			End
		
			index += 1

		End
		
		Return newCollection
		
	End

	Method Sum:T(collection:TCollection, transform:NumericTransform<T>) Where T Extends INumeric

		' Floating point or integer type?
		Local value:T = 1.5
		If String(value).Contains(".")
			Return CalculateSum<Double>(collection, transform)
		Else
			Return CalculateSum<Long>(collection, transform)
		End

	End		

	Method Take:TCollection(collection:TCollection, count:Int)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection

		Local index:UInt = 1
		
		For Local item := EachIn collection.All()
			If index <= count
				newCollection.Add(item)
				index += 1
			Else
				Return newCollection
			End
		End

		Return newCollection ' will get here if count > Self.Count()
		
	End

	Method TakeWhile:TCollection(collection:TCollection, predicate:Predicate<T>)
		Return TakeWhile(collection, Lambda:Bool(item:T, index:Int)
			Return predicate(item)
		End)
	End

	Method TakeWhile:TCollection(collection:TCollection, predicate:PredicateWithIndex<T>)
		
		Local newCollection := New TCollection()
		If collection.Empty Then Return newCollection

		Local index:UInt = 0
		
		For Local item := EachIn collection.All()
		
			If predicate(item, index)
				newCollection.Add(item)
			Else
				Return newCollection
			End
		
			index += 1

		End
		
		Return newCollection ' will get here if predicate is always true
		
	End

	Method ToMap<T,TKey>:Map<TKey,T>(collection:TCollection, keySelector:Selector<T,TKey>)
		
		Local map := New Map<TKey,T>()
		If collection.Empty Then Return map
		
		For Local item := EachIn collection.All()
			map.Add(keySelector(item), item)
		End
		
		Return map
		
	End

	Method ToMap<T,TKey,TElement>:Map<TKey,TElement>(collection:TCollection, keySelector:Selector<T,TKey>, elementSelector:Selector<T,TElement>)
		
		Local map := New Map<TKey,TElement>()
		If collection.Empty Then Return map
		
		For Local item := EachIn collection.All()
			map.Add(keySelector(item), elementSelector(item))
		End
		
		Return map
		
	End

	Method Union:TCollection(collection:TCollection, list:TCollection, comparer:Comparer<T>)
		
		Local newCollection := New TCollection(collection)

		If list.Empty Then Return newCollection
	
		For Local item := EachIn list.All()
			If Not Contains(newCollection, item, comparer) Then newCollection.Add(item)
		End
		
		Return newCollection
		
	End
						
Private

	Method CalculateAverage<N>:T(collection:TCollection, transform:NumericTransform<T>) Where N Extends INumeric
	
		Local total:N = 0

		For Local item := EachIn collection.All()

			If transform <> Null
				total += transform(item)
			Else
				total += item
			End

		End
		
		Return total / collection.Count()
		
	End

	Method CalculateMax<N>:T(collection:TCollection, transform:NumericTransform<T>, startValue:N) Where N Extends INumeric
	
		Local max:N = startValue

		For Local item := EachIn collection.All()

			Local value:N = item

			If transform <> Null
				value = transform(item)
			End
			
			If value > max
				max = value
			End

		End
		
		Return max
		
	End

	Method CalculateMin<N>:T(collection:TCollection, transform:NumericTransform<T>, startValue:N) Where N Extends INumeric
	
		Local min:N = startValue

		For Local item := EachIn collection.All()

			Local value:N = item

			If transform <> Null
				value = transform(item)
			End
			
			If value < min
				min = value
			End

		End
		
		Return min
		
	End

	Method CalculateSum<N>:T(collection:TCollection, transform:NumericTransform<T>) Where N Extends INumeric
	
		Local sum:N = 0

		For Local item := EachIn collection.All()

			Local value:N = item

			If transform <> Null
				value = transform(item)
			End
			
			sum += value

		End
		
		Return sum
		
	End

End
