Namespace jplib
#import "<std>"
#Import "<mojo>"

Using std..
Using mojo..

Global displayHitBox:Int = False
Global ATR:Float = Pi/180.0 
Global RTA:Float = 180.0/Pi

'***************************************************
'			graphics								*
'***************************************************

Global gfx:Graphics = New Graphics()
 
Class Graphics
	Field startX:Float
	Field startY:Float
	Field currentX:Float
	Field currentY:Float
	Field lastX:Float
	Field lastY:Float
	Field image:Image
	
	method New()
	End Method
	
	Method SetTexture(image:Image)
		Self.image = image
	End Method
	
 
	Method MoveTo(x:Float,y:Float)
		startX = x
		startY = y
		lastX = x
		lastY = y
	End Method
  
	Method LineTo(canvas:Canvas,x:Float,y:Float)
		currentX = x
		currentY = y
		canvas.DrawLine(lastX,lastY,x,y)
		lastX=x
		lastY=y
	End
	
	Method GLowLineTo(canvas:Canvas,x:Float,y:Float,width:Float = 3)
		DrawGlowLine(canvas,image,lastX,lastY,startX,startY,width)
		lastX = startX
		lastY = startY			
	End Method
  
	Method Close(canvas:Canvas)
		canvas.DrawLine(lastX,lastY,startX,startY)
		lastX = startX
		lastY = startY
	End Method
  
	Method Poly(canvas:Canvas,vert:Float[])
		If vert.Length < 6 Return
		MoveTo(vert[0],vert[1])
		For Local i:Int = 2 Until (vert.Length - 2) Step 2
			LineTo(canvas,vert[i+2],  vert[i+3])
		Next
		Close(canvas)
	End Method
  
	Method Circle:Void(canvas:Canvas,x_center:Float,y_center:Float,radius:Float,stps:Int = 40,filled:Int = False)
		
		Local angle:Float = 0.0
		Local arcs:= new Float[stps*2+2]
		' distance between each point
		Local stp:Float = 360.0 / stps
		For Local indx:Float = 0 To stps*2 Step 2 
			arcs[indx  ] = Cos(angle*ATR) * radius
			arcs[indx+1] = Sin(angle*ATR) * radius
			angle += stp
    	Next
		If filled = False		
    		MoveTo(x_center+arcs[0],y_center+arcs[1])
			For Local indx:Float = 2 until stps*2-4 Step 2 
				gfx.LineTo(canvas,x_center + arcs[indx  ], y_center + arcs[indx+1])
     		Next
    		Close(canvas)
		Else
			canvas.DrawPoly(arcs)
		Endif
	End Method

	Method Arc(canvas:Canvas,x:Float, y:Float, startAngle:Float, endAngle:Float, radius:Float,stpn:Float = 1.0)
		Local fx:Float,fy:Float 
		Local lx:Float,ly:Float 
		Const RATE:Float = Pi/180.0
		If startAngle = endAngle Then Return
		If startAngle > endAngle
			Local ta:Float = endAngle
			endAngle = startAngle
			startAngle = ta
		Endif
		Local angle:Float = endAngle - startAngle
		If angle > 360.0 angle = 360.0
		
		Local stp:Float = 1.0/(RATE * radius)
		Local accumAngle:Float = startAngle
	
		lx = Cos(accumAngle) * radius
		ly = Sin(accumAngle) * radius
		MoveTo(x+lx,y+ly)
		While accumAngle < (startAngle+angle)			
			lx = Cos((accumAngle)) * radius
			ly = Sin((accumAngle)) * radius
			LineTo(canvas, x + lx, y + ly)
			accumAngle += stp*stpn
		Wend

	End Method
 
 
	Method Ellipse (canvas:Canvas,x_center:Float, y_center:Float, w:Float, h:Float,angle:Float=0,n:Float=1)
		
		Local tx:Float
		Local cs:Float = Cos(angle)
		Local sn:Float = Sin(angle)
 		
 		Local angle_increment:Float = Pi / n
		Local theta:Float = 0.0
		
		Local vx:Float = w/2 * Cos (theta)
		Local vy:Float = h/2 * Sin (theta)
		tx = vx * cs - vy*sn
		vy = vx * sn + vy*cs
		MoveTo(x_center+tx,y_center+vy)
		theta += angle_increment
		While theta < 360.0
			vx = w/2 * Cos (theta)
			vy = h/2 * Sin (theta)
			tx = vx * cs - vy*sn
			vy = vx * sn + vy*cs
			LineTo(canvas,tx + x_center,vy + y_center)
			theta += angle_increment * 2
		Wend
		Close(canvas)
	End Method
	
	Method Rotate(angle:Float)
		Local cs:Float = Cos(angle)
		Local sn:Float = Sin(angle)
		
		Local tx:Float
		tx = startX * cs - startY * sn
		startY = startX * sn + startY * cs
		startX = tx
	End Method
	
	Method Rectangle(canvas:Canvas,x:Float,y:Float,width:Float,height:Float,filled:Bool=False)
		If filled
			Local vert:Float[] = New Float[8]
			vert[0] = x
			vert[1] = y
			vert[2] = x+width
			vert[3] = y
			vert[4] = x+width
			vert[5] = y+height
			vert[6] = x
			vert[7] = y+height
			Poly(canvas,vert)
		Else
			MoveTo(x,y)
			LineTo(canvas,x+width,y)
			LineTo(canvas,x+width,y+height)
			LineTo(canvas,x,y+height)
			Close(canvas)
		Endif
	End Method
	
End Class

Function TileImageAcross:Void(canvas:Canvas,image:Image,x:Float,y:Float,width:Float)
	Local w:Float = width-1
	Local ox:Float = -w+1
	Local px:Float = x
	Local fx:Float = px - Int(px)
	Local tx:Float = Int(px) - ox
	

	If (tx >= 0)
		tx=tx Mod w + ox
	Else
		tx=w + tx Mod w + ox
	Endif
	Local vr:Float = 680.0
	Local ix:Float = tx
	While(ix < vr)
		canvas.DrawImage(image,ix+fx,y)
		ix=ix+w
	Wend
End Function

Function GenerateGlowImg:Image()
	Local img := New Image(1,16)
	Local canvas := New Canvas(img)
	canvas.Clear( New Color(0,0,0,0))
	canvas.Color = New Color(.9,.9,.9,.9)
	canvas.DrawPoint(0,8)
	canvas.DrawPoint(0,7)
	canvas.Color = New Color(.8,.8,.8,.7)
	canvas.DrawPoint(0,9)
	canvas.DrawPoint(0,6)
	canvas.Color = New Color(.7,.7,.7,.6)
	canvas.DrawPoint(0,10)
	canvas.DrawPoint(0,5)
	canvas.Color = New Color(.6,.6,.6,.5)
	canvas.DrawPoint(0,11)
	canvas.DrawPoint(0,4)
	canvas.Color = New Color(.5,.5,.5,.4)
	canvas.DrawPoint(0,12)
	canvas.DrawPoint(0,3)
	canvas.Color = New Color(.4,.4,.4,.3)
	canvas.DrawPoint(0,13)
	canvas.DrawPoint(0,2)
	canvas.Color = New Color(.3,.3,.3,.2)
	canvas.DrawPoint(0,14)
	canvas.DrawPoint(0,1)
	canvas.Color = New Color(.2,.2,.2,.1)
	canvas.DrawPoint(0,15)
	canvas.DrawPoint(0,0)
	canvas.Flush()
	Return img
End Function

Function DrawGlowLine(canvas:Canvas,image:Image,x1:Float,y1:Float,x2:Float,y2:Float,width:Float = 3)
	Local vx:Float = x2-x1
	Local vy:Float = y2-y1
	Local length:Float = Sqrt(vx*vx+vy*vy)
	If length = 0 Return
	Local ang:Float = ATan2(vy,vx)
	Local dx:Float = Cos(ang)
	Local dy:Float = Sin(ang)
	vx = -dy * width/2
	vy = dx * width/2
	canvas.PushMatrix()
	canvas.Translate(x1-vx,y1-vy)
	canvas.Rotate(-ang)
	canvas.DrawRect(New Rectf(0,0,length,width),image)
	canvas.PopMatrix()
End Function

'glowLine Texture
 Function TextureImage:Image()
    Local img:Image = New Image(1,5)
	Local icanvas := New Canvas(img)
	icanvas.Clear(New Color(0,0,0,0))
    icanvas.Color = New Color(1,1,1,.3)
	icanvas.DrawPoint(0,0)
	icanvas.Color = New Color(1,1,1,.6)
	icanvas.DrawPoint(0,1)
	icanvas.Color = New Color(1,1,1,1)
	icanvas.DrawPoint(0,2)
	icanvas.Color = New Color(1,1,1,.6)
	icanvas.DrawPoint(0,3)
	icanvas.Color = New Color(1,1,1,.3)
	icanvas.DrawPoint(0,4)
	icanvas.Flush()
	img.Handle = New Vec2f(0,.5)
	Return img 
End Function
	  
class Bezier
	Field x1:Float,y1:Float,x2:Float,y2:Float,x3:Float,y3:Float,x4:Float,y4:Float
	Field inc:Float
	
	Method New(x1:Float,y1:Float,x2:Float,y2:Float,x3:Float,y3:Float,x4:Float,y4:Float,inc:Float=-1)
		Self.x1=x1
		Self.y1=y1
		Self.x2=x2
		Self.y2=y2
		Self.x3=x3
		Self.y3=y3
		Self.x4=x4
		Self.y4=y4
		
		If inc=-1
			Local dx:Float=x4-x1
			Local dy:Float=y4-y1
			Local d:Float=Sqrt(dx*dx+dy*dy)
			Local inc:Int=1/d
		Endif
		self.inc=inc
		
	End Method
	
	Class Point
		Field x:Float
		Field y:float
		
		Method New()
		End Method
		
		Method New(x:Float,y:Float)
			Self.x = x
			Self.y = y
		End Method
	End Class
	
	Struct Iterator
		Field b:Bezier
		Field t:Float
		Field point:Point
		
		Method New(b:Bezier)
			Self.b=b
			point = New Point()
		End Method
		
		Property AtEnd:Bool()
			return t<=1
		End
	
		Method Bump()
			b.pos(t,point)
			If t=1
				t += 1
				point.x = b.x4
				point.y = b.y4
			Endif
			t += b.inc
			If t>1 t=1
		End Method
	End Struct

	Method objectenumerator:Iterator()
		Return New Iterator(Self)
	End Method
	
	Method pos(t:Float,point:Point)
		Local nt:Float=1-t
		point.x = nt*nt*nt*x1 + 3*nt*nt*t*x2 + 3*nt*t*t*x3 + t*t*t*x4
		point.y = nt*nt*nt*y1 + 3*nt*nt*t*y2 + 3*nt*t*t*y3 + t*t*t*y4
	End Method
End Class


Function test()
	Local b := New Bezier( 0,0, 100,100, 200,200, 300,0 )

	'you can use EachIn to get all the points along the curve
'	Local point := New Bezier.Point()

'	For point= EachIn b
'		DrawRect(point.x,point.y,1,1)
'	Next

	'or you can do it directly
'	Local p := New Bezier.Point
'	Local t:Float=0
'
'	For t=0 to 1 Step 0.01
'		b.pos(t,p)
'		DrawRect(p.x,p.y,1,1)
'	Next
	
End Function

Function PointInOval:Int(px:float,py:float,ox:float,oy:float,width:float,height:Float)

	Return((px - ox)*(px - ox) / width*width + (py - oy)*(py - oy) / height*height < 1)

End Function
