
#Import "<libc>"
#Import "<std>"
#Import "<mojo>"

Using std..
Using mojo..
Using libc..

Function Main()

	New AppInstance
	
	New ParticleMapTester(1024, 768)
	
	App.Run()
End

'------

Class ParticleMapTester Extends Window

	Field imgGreyscale:Image
	Field imgColorBlended:Image
	Field imgColorDiscrete:Image
	
	Field hmWidth:Int = 256
	Field hmHeight:Int = 256
	Field particles:UInt
	Field clusters:UInt
	Field strSticky:String

	Field data:Double[,]

	Method New(width:Int, height:Int)

		Super.New("Particle Algorithm Test", width, height)
		
		ClearColor = Color.Black
		SwapInterval = 0
		
		local tv:timeval
		gettimeofday(varptr(tv))
		SeedRnd(tv.tv_sec + tv.tv_usec)

		data = New Double[hmWidth, hmHeight]

		Generate()
	End
	
	Method Generate:Void()
	
		particles = Rnd(250, 5000)
		clusters = Rnd(1, 500)
		Local sticky := Cast<Bool>(Floor(Rnd(0,2)))
		strSticky = sticky ? "Style: Sticky" Else "Style: Rolling"
		
		GenerateParticleMap(data, particles, clusters, sticky)
		
		imgGreyscale = New Image(CreateGreyscalePixmap(data))
		imgColorBlended = New Image(CreateColorPixmap(data))
		imgColorDiscrete =  New Image(CreateColorPixmap(data, , True))
		
		GCCollect()

	End

	Method OnWindowEvent(event:WindowEvent) Override
		If event.Type = EventType.WindowClose
			App.Terminate()		
		Endif
	End
	
	Method OnMouseEvent( event:MouseEvent ) Override
		If event.Type = EventType.MouseDown
			If event.Button = 1
				Generate()
			Endif				
		EndIf
	End
	
	Method OnRender(canvas:Canvas) Override
	
		App.RequestRender()

		canvas.Color = Color.White
		
		canvas.DrawText("Left Click to generate new particle map." , 10, 10)
		canvas.DrawText("particles: " + particles + ", clusters " + clusters + ", " + strSticky, 10, canvas.Font.Height * 2 + 10)
		canvas.DrawImage(imgGreyscale, 10, canvas.Font.Height * 3 + 30)
		canvas.DrawImage(imgColorBlended, hmWidth + 11, canvas.Font.Height * 3 + 30)
		canvas.DrawImage(imgColorDiscrete, hmWidth * 2 + 12, canvas.Font.Height * 3 + 30)

		canvas.Flush()

	End
	
End


'------------ 


Function GenerateParticleMap:Void(array:Double[,], particles:Int, clusters:Int, sticky:Bool = True, amts:Int[] = Null)
 	'Generate a heightmap in a 2d array using the "particle" algorithms.
 	'particles: number of particles to calculate per cluster. Minimum 1.
  	'clusters: number of clusters: higher number gives greater map density. Minimum 1.
	'sticky: algorithm variation: True = sticky, False = rolling.
	'amts: algorithm variation. Should be an integer array of 4 values. Suggested range -5 to 5.

	If Not array Then Return
	
	Local width := array.GetSize(0)
	Local height := array.GetSize(1)
		
	For Local x := 0 Until width
		For Local y := 0 Until height
			array[x, y] = 0.0
		Next
	Next
		
	If amts = Null Or amts.Length < 4 Then amts = New Int[](-1, 1, -1, 1)
	
	Local disp:Float = 1.0

	If sticky 					'Sticky style

    	For Local m := 1 To clusters
    	
	 		Local x := Floor(Rnd(0, width)) 
      		Local y := Floor(Rnd(0, height))

	    	For Local n := 1 To particles
		  		
				If (((x < width) And (y < height) And (x >= 0) And (y >= 0)))
					
					array[x, y] += disp

			    	Select Floor(Rnd(0, amts.Length))
     		   			Case 0
	          				x += amts[0]
							
    	     			Case 1
	     	       			x += amts[1]
								
						Case 2
							y += amts[2]
		
        		  		Case 3
     	       				y += amts[3]
        			End
				
				Endif
			Next
    	Next
	
	Else 						'Rolling style

	    For Local m := 1 To clusters
		 	 	
		 	Local x := Floor(Rnd(0, width)) 
	      	Local y := Floor(Rnd(0, height))
	
		    For Local n := 1 To particles
	
	     	   	If (((x < width) And (y < height) And (x >= 0) And (y >= 0)))
			
	      			Local h := array[x, y] + disp
	      			array[x, y] = h
	
		      		Local startx := x - 1
	     	     	If (startx < 0) Then startx = 0
	
	          		Local endx := x + 1
	          		If (endx > width) Then endx = width
	
	          		Local starty := y - 1
	     	     	If (starty < 0) Then starty = 0
	
	     	    	Local endy := y + 1
	          		If (endy > height) Then endy = height
				
	     	     	For Local x2 := startx Until endx
						For Local y2 := starty Until endy
							Local h2 := array[x2, y2]
	              			If (h2 < h) Then array[x2, y2] += disp
	            		Next
					Next
	
	    		Endif

			    Select Floor(Rnd(0, amts.Length))
	     		    Case 0
		           		x += amts[0]
								
	    	     	Case 1
		     	   		x += amts[1]
									
					Case 2
						y += amts[2]
			
	        		Case 3
	     	    		y += amts[3]
	        	End
					
			Next
	
	    Next
	
	Endif

	'''Normalise
	Local minv := array[0, 0]
	Local maxv := array[0, 0]   	
		   	
	For Local x := 0 Until width
		For Local y:= 0 Until height
			If (array[x, y] < minv)
				minv = array[x, y]
			Else 
				If (array[x, y] > maxv) Then maxv = array[x, y]
			Endif
		Next
	Next
			
	For Local x := 0 Until width
		For Local y:= 0 Until height
			array[x, y] = (array[x, y] - minv) / (maxv - minv)
		Next
	Next
	'''
    
End


'------------ 


Function CreateGreyscalePixmap:Pixmap(array:Double[,])
		If Not array Then Return Null
		
		Local width := array.GetSize(0)
		Local height := array.GetSize(1)
		Local pm := New Pixmap(width, height, PixelFormat.RGB24)
				
	    For Local x := 0 Until width
	     	For Local y := 0 Until height
	          	Local v:UInt = Floor(array[x, y] * 255.0)
	         	pm.SetPixel(x, y, Color.FromARGB(v Shl 16 | v Shl 8 | v Shl 0))
	      	Next
	    Next
		
		Return pm
End

Function CreateColorPixmap:Pixmap(array:Double[,], colarr:Color[] = Null, discrete:Bool = false)
		If Not array Then Return Null

		Local width := array.GetSize(0)
		Local height := array.GetSize(1)
		Local pm := New Pixmap(width, height, PixelFormat.RGB24)
	
		If (colarr = Null) Or (colarr.Length <= 1)
			colarr = New Color[](Color.Blue, Color.Skin, Color.Green, Color.Brown, Color.DarkGrey, Color.Grey)		
		Endif
			
		Local div:Double = 1.0 / Double(colarr.Length)
		
		If discrete
			For Local x := 0 Until width 
  				For Local y := 0 Until height 
      				For Local i := 0 Until colarr.Length
						Local lvl:Double = Double(i) * div
						If array[x, y] >= lvl Then pm.SetPixel(x, y, colarr[i])
					Next
				Next
			Next
		Else
			For Local x := 0 Until width
   				For Local y := 0 Until height
   				
					If array[x, y] > (1.0 - div) 
						pm.SetPixel(x, y, colarr[colarr.Length - 1])
					Else
						Local val:Double = 0.0
						Local color := New Color(0.0, 0.0, 0.0)		

						For Local i:Int = 0 until colarr.Length
										
							Local clr := New Color(colarr[i].R, colarr[i].G, colarr[i].B)				
							
							Local factor:Double = (div - Abs(val - array[x, y])) / div
							If factor < 0.0 Then factor = 0.0 Else If factor > 1.0 Then factor = 1.0
										
							color.R += (clr.R * factor)	
							color.G += (clr.G * factor)
							color.B += (clr.B * factor)			
		
							val += div
							
						Next
	
						pm.SetPixel(x, y, color)
		
					EndIf

				Next
			Next
		Endif
				
		Return pm		
End


