Namespace mojo3d.graphics


Class AnimSprite Extends Sprite

	Field onLastFrame :Void()
	Field onFirstFrame :Void()
	
	Protected
	
	'Atlas values
	
	Field _frame:Int										'The frame currently displayed
	Field _coordinates:= New Stack<Rect<Double>>			'A stack containing the UV coordinates for each cell
	Field _rows:Int											'Number of rows in the original image file
	Field _columns:Int										'Number of collumns in the original image file
	Field _cellWidth:Double									'The width of an individual cell (frame or tile), in pixels
	Field _cellHeight:Double								'The height of an individual cell (frame or tile), in pixels
	Field _padding :Double									'the gap between cells, in pixels
	Field _border :Double									'the gap between the texture's edges and the cells, in pixels
	Field _paddedWidth:Double								'the total width of a cell + padding, in pixels
	Field _paddedHeight:Double								'the total height of a cell + padding, in pixels
	
	'Animation clip management
	Field _timeScale := 1.0									'Adjusts playback speed. Can be used to create slow motion effects without changing the framerate.
	Field _anim :AnimationClip								'The animation clip currently playing
	Field _animations := New StringMap< AnimationClip >		'List of available animation clips
	Field _offset :Int										'Current frame offset
	Field _framerate:Double	= 15.0							'Frame rate, if a clip is not provided
	
	'Misc
	Field _firstFrame := True
	Field _hasReachedEnd := False
	Field _hasReachedStart := False
	
	'******************************* Public Properties *******************************
	Public
	
	'Current frame
	Property Frame:Int()		
		Return _frame
	Setter( number:Int )
		_frame = Clamp( number, 0, _coordinates.Length-1 )
		TextureRect = _coordinates[ _frame ]
	End
	
	
	'current AnimationClip
	Property Animation:String()		
		Return _anim.name
	Setter( name:String )
		If _animations[ name ]
			_anim = _animations[ name ]
		Else
			Print( "AnimSprite: animation '" + name + "' Not found!" )
		End
	End
	
	
	'Map with all animations as values and their names as keys
	Property AllAnimations:StringMap< AnimationClip >()
		Return _animations
	End
	
	
	'Offsets the current frame, useful when creating multiple copies of the same sprite
	Property FrameOffset:Int()
		Return _offset
	Setter( number:Int )
		_offset = number
	End
	
	
	'Sets the frame rate. If a clip is in use, that clip's frame rate will be set.
	Property FrameRate:Double()
		If _anim
			Return _anim.framerate
		Else
			Return _framerate
		End
	Setter( value:Double )
		If _anim
			_anim.framerate = value
		Else
			_framerate = value
		End
	End
	
	
	'Duration (in seconds) of the current animation clip. If none, duration of the entire spritesheet is provided.
	Property Duration:Double()
		Local _period:Double = ( 1.0 / FrameRate ) * _timeScale
		If _anim
			Return _period * _anim.frames.Length
		Else
			Return _period * _coordinates.Length
		End
	End


	Property FirstFrame:Int()
		If _anim
			Return _anim.frames[ 0 ]
		Else
			Return 0
		End	
	End
	
	
	Property LastFrame:Int()
		If _anim
			Return _anim.frames[ _anim.frames.Length-1 ]
		Else
			Return _coordinates.Length - 1
		End	
	End
	
	
	'******************************* Public Methods *******************************
	
	
	Method New( path:String, cellWidth:Int, cellHeight:Int, padding:Int = 0, border:Int = 0, flags:TextureFlags = TextureFlags.FilterMipmap )
		'Creates new Sprite with material
		Super.New( SpriteMaterial.Load( path, flags ) )
		Local texture := Cast<SpriteMaterial>( Material ).ColorTexture
		'Set various atlas related fields
		_padding = padding
		_border = border
		_cellWidth = cellWidth
		_cellHeight = cellHeight
		_paddedWidth = _cellWidth + ( _padding * 2 )
		_paddedHeight = _cellHeight + ( _padding * 2 )
		_rows = ( texture.Height - _border - _border ) / _paddedHeight
		_columns = ( texture.Width - _border - _border ) / _paddedWidth
		'Generates UV coordinates for each frame
		Local numFrames := _rows * _columns
		Local w:Double = texture.Width
		Local h:Double = texture.Height
		For Local i:= 0 Until numFrames
			Local col := i Mod _columns
			Local x:Double = ( col * _paddedWidth ) + _padding + _border
			Local y:Double = ( ( i / _columns ) * _paddedHeight ) + padding + border
			_coordinates.Push( New Rectf( x/w, y/h, (x+_cellWidth)/w, (y+_cellHeight)/h ) )
		Next
		'Defaults to frame 0
		Frame = 0
	End
	
	
	'Update time is in seconds (Double), not milliseconds
	Method Update( time:Double )
		Local frameLength:Double = ( 1.0 / FrameRate ) / _timeScale
		time += ( _offset * frameLength )
		
		If _anim
			If _anim.loop
				Frame = _anim.frames[ Int( ( time Mod Duration ) / frameLength ) ]
			Else
				Frame = _anim.frames[ Int( time / frameLength ) ]
			End
		Else
			Frame = Int( ( time Mod Duration ) /frameLength ) 
		End
		
		If _frame = LastFrame
			If Not _hasReachedEnd
				onLastFrame()
				_hasReachedEnd = True
				_hasReachedStart = False
			End		
		End
		
		If _firstFrame
			onFirstFrame()
			_firstFrame = False
		Else
			If _frame = FirstFrame
				If Not _hasReachedStart
					onFirstFrame()
					_hasReachedStart = True
					_hasReachedEnd = False
				End		
			End		
		End
		
	End
	
	
	'Adds new animation clips.
	Method AddAnimationClip( _name:String, _loop:Bool = True, framerate:Int, _frames:Int[] )
		local animClip := New AnimationClip
		animClip.name = _name
		animClip.loop = _loop
		animClip.frames = _frames
		animClip.framerate = framerate
		_animations.Add( _name, animClip )
	End


	'Loads Json file
	Method LoadAnimations( path:String )
		Local json := JsonObject.Load( path )
		If json
			Local anims := json.GetObject( "animations" )
			If anims
				For Local a := Eachin anims.ToObject()
					Local obj := a.Value.ToObject()
					Local loop := obj[ "loop" ].ToBool()
					Local rate := obj[ "rate" ].ToNumber()

					Local frameStack := obj[ "frames" ].ToArray()
					Local frames:Int[] = New Int[ frameStack.Length ]
					For Local n := 0 Until frameStack.Length
						frames[n] = frameStack[n].ToNumber()
					Next

					AddAnimationClip( a.Key, loop, rate, frames )
				Next
			End
		Else
			Print( "AnimSprite: Warning, json file not found or invalid" )
		End
	End
	
End


'**************************************************************************************************************************************'


Class AnimationClip						'AnimationClips contain a sequence of frames to be played, its name, framerate and loop state.
	
	field name			:String			'Animation name
	field loop			:= True			'looping can be controlled per animation
	field frames 		:Int[]			'Frame list array, contains the sequence in which the frames play
	Field framerate		:= 15.0			'frame rate per clip

	Property FrameCount:Int()
		Return frames.Length
	End
	
End
