
Namespace datapacker

Using stb.image
Using std.stream


Class Packer
	
	Method New( packFile:String )
		
		_packFile=packFile
	End
	
	Method Pack( files:String[],names:String[]=Null )
		
		If names=Null Then names=files
		
		Local stream:=FileStream.Open( _packFile,"w" )
		Local offset:=0
		Local json:=New JsonObject
		
		For Local i:=0 Until files.Length
			Local file:=files[i]
			Local b:=DataBuffer.Load( file )
			Assert( b<>Null,"Can't load resource to pack: "+file )
			
			Local size:=b.Length
			stream.Write( b.Data,size )
			
			Local j:=New JsonObject
			json[names[i]]=j
			j["offset"]=New JsonNumber( offset )
			j["size"]=New JsonNumber( size )
			
			offset+=b.Length
		Next
		
		stream.Close()
		
		SaveString( json.ToJson(),_packFile+".json" )
	End
	
	Method Load( packFile:String )
	
		_packFile=packFile
		_stream=FileStream.Open( _packFile,"r" )
		_json=JsonObject.Load( _packFile+".json" )
	End
	
	Method GetImage:Image( name:String )
		
		Local offset:Int,size:Int
		
		GetParams( name,Varptr offset,Varptr size )
		
		Local img:=Image.Load( _stream,offset,size )
		Return img
	End
	
	Method Close()
		
		If _stream Then _stream.Close()
		_stream=Null
	End
	
	Private
	
	Field _packFile:String
	Field _stream:Stream
	Field _json:JsonObject
	
	Method GetParams( name:String,offset:Int Ptr,size:Int Ptr )
		
		If Not _stream Then Load( _packFile )
		
		DebugAssert( _stream<>Null,"Stream isn't opened! Call to Load() before GetX..()." )
		
		If _json.Contains( name )
			Local j:=_json[name].ToObject()
			offset[0]=j["offset"].ToNumber()
			size[0]=j["size"].ToNumber()
		Endif
	End
	
End

Class Image Extension
	
	Function Load:Image( stream:Stream,offset:Int,size:Int,textureFlags:TextureFlags=TextureFlags.FilterMipmap,shader:Shader=Null )
		
		Local pixmap:=LoadPixmap( stream,offset,size )
		If Not pixmap Return Null
		
		pixmap.PremultiplyAlpha()
		
		Local image:=New Image( pixmap,textureFlags,shader )
		
		Return image
	End
	
End


Private

Struct stbi_user
	Field stream:Stream
	Field eof:Int
End

Function stbi_read:Int( user:Void Ptr,data:stbi_char Ptr,count:Int )
	Local stream:=Cast<stbi_user Ptr>( user )[0].stream
	Return stream.Read( data,count )
End

Function stbi_skip:Void( user:Void Ptr,count:Int )
	Local stream:=Cast<stbi_user Ptr>( user )[0].stream
	stream.Seek( stream.Position+count )
End

Function stbi_eof:Int( user:Void Ptr )
	Local u:=Cast<stbi_user Ptr>( user )[0]
	Return u.eof
End

Class StbPixmap Extends Pixmap
	
	Method New( width:Int,height:Int,format:PixelFormat,data:UByte Ptr,pitch:Int )
		Super.New( width,height,format,data,pitch )
		
		_data=data
	End
	
	Private
	
	Field _data:UByte Ptr
	
	Method OnDiscard() Override
		
		Super.OnDiscard()
		
		stbi_image_free( _data )
		
		_data=Null
	End
	
	Method OnFinalize() Override
		
	 	stbi_image_free( _data )
	End
End

Function LoadPixmap:Pixmap( stream:Stream,offset:Int,size:Int,format:PixelFormat=PixelFormat.Unknown )

	Local x:Int,y:Int,comp:Int,req_comp:Int
	
	If format<>PixelFormat.Unknown req_comp=PixelFormatDepth( format )
	
	stream.Seek( offset )
	
	Local user:stbi_user
	user.stream=stream
	user.eof=Min( offset+size,stream.Length )
	
	Local clbks:stbi_io_callbacks
	clbks.read=stbi_read
	clbks.skip=stbi_skip
	clbks.eof=stbi_eof
	
	Local data:=stbi_load_from_callbacks( Varptr clbks,Varptr user,Varptr x,Varptr y,Varptr comp,req_comp )
	
	stream.Close()
	
	If Not data Print "not data, "+stream.Length ; Return Null
	
	If format=PixelFormat.Unknown
		Select comp
		Case 1 format=PixelFormat.I8
		Case 2 format=PixelFormat.IA16
		Case 3 format=PixelFormat.RGB24
		Case 4 format=PixelFormat.RGBA32
		Default Assert( False )
		End
	End
	
	Local pixmap:=New StbPixmap( x,y,format,data,x*PixelFormatDepth( format ) )
	
	Return pixmap
End
